from flask import Flask
from flask_sqlalchemy import SQLAlchemy
# 1. IMPORT Flask-Login
from flask_login import UserMixin, LoginManager, login_required, current_user
import os
import subprocess 
import logging

# Set logging level for SQLAlchemy/Flask to avoid excessive output
logging.basicConfig()

# --- App & DB Setup ---
app = Flask(__name__)

# Import config settings for file paths
from config import UPLOAD_FOLDER, TRANSCRIPT_FOLDER

app.config['UPLOAD_FOLDER'] = UPLOAD_FOLDER
app.config['TRANSCRIPT_FOLDER'] = TRANSCRIPT_FOLDER
app.config['SQLALCHEMY_DATABASE_URI'] = 'sqlite:///database.db'
app.config['SECRET_KEY'] = 'a-super-secret-key-for-flash' # Required for flash messages

db = SQLAlchemy(app)

# 2. FLASK-LOGIN SETUP
login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'login' # Name of the login route function

@login_manager.user_loader
def load_user(user_id):
    # Flask-Login requires a function to load a user from the database
    return User.query.get(int(user_id))

# --- Database Models ---

# 3. USER MODEL (with UserMixin for Flask-Login features)
class User(db.Model, UserMixin):
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(64), unique=True, nullable=False)
    password_hash = db.Column(db.String(128), nullable=False) # Store a hashed password

class Recording(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    filename = db.Column(db.String(255))
    transcript = db.Column(db.Text)
    status = db.Column(db.String(50)) # e.g., 'pending', 'processing', 'done', 'error'
    model_key = db.Column(db.String(100), nullable=True) # Column to track model used

with app.app_context():
    # This will create the database file and table if it doesn't exist (including the new User table)
    db.create_all()

# --- Import and Register Routes ---
import routes 
# CRITICAL: Pass app, db, Recording, AND User to the registration function
routes.register_routes(app, db, Recording, User) 


if __name__ == '__main__':
    # Use a specific port for deployment environment compatibility
    app.run(host='0.0.0.0', port=8090, debug=True)