import os
from flask import flash

# We rely on components being passed as arguments to maintain the circular dependency fix pattern.
def clean_zombie_entries(app_context, db_context, Recording_model, UPLOAD_FOLDER, TRANSCRIPT_FOLDER):
    """
    Checks all Recording entries against the filesystem.
    If the primary audio file is missing, the entry is considered a 'zombie'
    and is deleted along with any corresponding transcript file.
    """
    deleted_count = 0
    
    # Must run database operations within the application context
    with app_context.app_context():
        print("--- Starting Zombie Cleanup Check ---")
        
        # Query all records
        all_recordings = db_context.session.execute(db_context.select(Recording_model)).scalars().all()
        
        for item in all_recordings:
            # Check for the main audio file
            audio_path = os.path.join(UPLOAD_FOLDER, item.filename)
            # Check for the potential transcript file
            transcript_path = os.path.join(TRANSCRIPT_FOLDER, item.filename + ".txt")

            # The record is a zombie if the main audio file is missing.
            if not os.path.exists(audio_path):
                print(f"ZOMBIE FOUND: ID {item.id}, Filename {item.filename}. Audio file missing.")
                
                # 1. Delete associated transcript file if it exists
                if os.path.exists(transcript_path):
                    try:
                        os.remove(transcript_path)
                        print(f"  -> Deleted lingering transcript file: {os.path.basename(transcript_path)}")
                    except Exception as e:
                        print(f"  -> WARNING: Failed to delete transcript file {os.path.basename(transcript_path)}: {e}")
                
                # 2. Delete database entry
                try:
                    db_context.session.delete(item)
                    deleted_count += 1
                except Exception as e:
                    print(f"  -> ERROR: Failed to delete DB entry ID {item.id}: {e}")
            
            # Additional check: Clean up transcript files for 'done' status recordings if DB entry is missing
            # This is not strictly a "zombie" entry cleanup but good filesystem hygiene. (Skipped for simplicity, focusing on the request)
        
        # Commit all deletions in one go
        try:
            db_context.session.commit()
        except Exception as e:
            db_context.session.rollback()
            print(f"ERROR: Failed to commit zombie cleanup changes: {e}")
            flash("Database error during cleanup. Changes were rolled back.", 'error')
            return 0
        
        print(f"--- Zombie Cleanup Complete: {deleted_count} entries deleted ---")
        
        # Use Flask's flash message for feedback on the web interface
        if deleted_count > 0:
            flash(f"Cleanup successful: {deleted_count} orphaned database entries and associated transcript files were deleted.", 'warning')
        else:
            flash("Cleanup successful: No orphaned database entries were found.", 'info')

        return deleted_count

# The if __name__ == '__main__': block remains to show it's not for standalone use.