import os
import time
import threading
from flask import request, redirect, url_for, render_template, flash, jsonify, send_from_directory, current_app
from werkzeug.utils import secure_filename
# 1. IMPORT LOGIN FUNCTIONS and PASSWORD TOOLS
from flask_login import login_user, logout_user, login_required, current_user
from werkzeug.security import generate_password_hash, check_password_hash 
from whisper_worker import transcribe_recording
from clean_zombies import clean_zombie_entries
# 2. IMPORT REGISTRATION_ENABLED
from config import UPLOAD_FOLDER, TRANSCRIPT_FOLDER, ALLOWED_EXTENSIONS, AVAILABLE_MODELS, WHISPER_UI_MODELS, DEFAULT_MODEL_KEY, REGISTRATION_ENABLED

# ------------------------------------------------
# Model Configuration: Use the defined UI models
# ------------------------------------------------
WHISPER_MODELS = WHISPER_UI_MODELS

# --- Helpers ---
def allowed_file(filename):
    """Checks if a file has an allowed extension."""
    return '.' in filename and filename.rsplit('.', 1)[1].lower() in ALLOWED_EXTENSIONS

def save_file(file):
    """Saves the uploaded file with a unique, timestamp-based filename."""
    # Use Unix timestamp + random component as filename to ensure uniqueness
    timestamp = int(time.time())
    unique_part = os.urandom(4).hex()
    ext = file.filename.rsplit('.', 1)[1].lower()
    filename = f"{timestamp}-{unique_part}.{ext}"
    path = os.path.join(UPLOAD_FOLDER, filename)
    file.save(path)
    return filename

def start_transcription_worker(rec_id, model_key, app_context, db_context, Recording_model):
    """Starts the transcription in a background thread."""
    # We pass the application context and DB/Model references to the worker thread
    thread = threading.Thread(target=transcribe_recording, args=(rec_id, model_key, db_context, Recording_model, app_context))
    thread.start()

# --- NEW: User Routes ---

def login(db, User):
    if current_user.is_authenticated:
        return redirect(url_for('upload_and_list'))
        
    if request.method == 'POST':
        username = request.form.get('username')
        password = request.form.get('password')
        
        # Use .scalar_one_or_none() for single result queries
        user = db.session.execute(db.select(User).filter_by(username=username)).scalar_one_or_none()
        
        if user and check_password_hash(user.password_hash, password):
            login_user(user)
            flash('Logged in successfully.', 'info')
            # Redirect to the page the user tried to access (next parameter) or the default list
            next_page = request.args.get('next')
            return redirect(next_page or url_for('upload_and_list'))
        else:
            flash('Invalid username or password.', 'error')
            
    # Pass the registration status to the template to show/hide the link
    return render_template('login.html', registration_enabled=REGISTRATION_ENABLED)

def logout():
    logout_user()
    flash('You have been logged out.', 'info')
    return redirect(url_for('login')) # Redirect to login page after logout

# --- Setup/Registration Route (Protected by REGISTRATION_ENABLED) ---
def register(db, User):
    # Security check: If registration is disabled, block access
    if not REGISTRATION_ENABLED:
        flash('User registration is currently disabled by the administrator.', 'error')
        return redirect(url_for('login')) 
        
    if current_user.is_authenticated:
        flash('Already logged in.', 'warning')
        return redirect(url_for('upload_and_list'))

    if request.method == 'POST':
        username = request.form['username']
        password = request.form['password']
        
        # Check if user already exists
        if db.session.execute(db.select(User).filter_by(username=username)).scalar_one_or_none():
            flash('User already exists.', 'error')
            return render_template('register.html')

        hashed_password = generate_password_hash(password)
        new_user = User(username=username, password_hash=hashed_password)
        db.session.add(new_user)
        db.session.commit()
        flash('User created successfully. Remember to disable registration in config.py.', 'info')
        return redirect(url_for('login'))
        
    return render_template('register.html')
# --------------------------------------------------------------------------

# --- Route Handlers (Apply @login_required decorator to all access-controlled routes) ---

@login_required 
def upload_and_list(db, Recording):
    # GET: Display list of recordings
    if request.method == 'GET':
        recordings = db.session.execute(db.select(Recording).order_by(Recording.id.desc())).scalars().all()
        return render_template('index.html', recordings=recordings, models=WHISPER_MODELS, default_model=DEFAULT_MODEL_KEY)
    
    # POST: Handle file upload and start worker
    if request.method == 'POST':
        if 'file' not in request.files:
            flash('No file part', 'error')
            return redirect(request.url)
        file = request.files['file']
        if file.filename == '':
            flash('No selected file', 'error')
            return redirect(request.url)
        if file and allowed_file(file.filename):
            try:
                filename = save_file(file)
                model_key = request.form.get('model_key', DEFAULT_MODEL_KEY)

                new_rec = Recording(filename=filename, status='pending', model_key=model_key)
                db.session.add(new_rec)
                db.session.commit()
                
                # FIX: Explicitly resolve current_app to the actual object before passing to thread
                app_instance = current_app._get_current_object()
                start_transcription_worker(new_rec.id, model_key, app_instance, db, Recording)
                
                flash(f'File {filename} uploaded. Transcription started with model: {model_key}.', 'success')
                return redirect(url_for('upload_and_list'))
            except Exception as e:
                db.session.rollback()
                flash(f'An error occurred during upload/start: {e}', 'error')
                return redirect(request.url)
        else:
            flash('File type not allowed.', 'error')
            return redirect(request.url)

@login_required 
def view_transcript(db, Recording, rec_id):
    recording = db.session.get(Recording, rec_id)
    if not recording:
        flash('Recording not found.', 'error')
        return redirect(url_for('upload_and_list'))
    
    # Check if the transcript is ready
    if recording.status != 'done' or not recording.transcript:
        flash(f'Transcript for ID {rec_id} is not ready yet. Status: {recording.status}.', 'warning')
        return redirect(url_for('upload_and_list'))

    # Construct the audio URL
    audio_url = url_for('serve_audio', rec_id=rec_id)
    
    return render_template('transcript.html', recording=recording, audio_url=audio_url)

@login_required 
def download_transcript(db, Recording, rec_id):
    recording = db.session.get(Recording, rec_id)
    if not recording or recording.status != 'done' or not recording.transcript:
        flash('Transcript not ready or recording not found.', 'error')
        return redirect(url_for('upload_and_list'))
    
    # The transcript file is stored as item.filename + .txt in TRANSCRIPT_FOLDER
    transcript_filename = recording.filename + ".txt"
    
    # Use Flask's safe way to serve files from a directory
    try:
        return send_from_directory(TRANSCRIPT_FOLDER, transcript_filename, as_attachment=True)
    except Exception as e:
        print(f"Error serving download: {e}")
        flash('Failed to find or serve the transcript file.', 'error')
        return redirect(url_for('view_transcript', rec_id=rec_id))

@login_required 
def serve_audio(db, Recording, rec_id):
    recording = db.session.get(Recording, rec_id)
    if not recording:
        return "Audio file not found.", 404
        
    # Send the audio file from the UPLOAD_FOLDER
    try:
        return send_from_directory(UPLOAD_FOLDER, recording.filename)
    except Exception:
        return "Error serving audio.", 500

@login_required 
def recording_status(db, Recording, rec_id):
    recording = db.session.get(Recording, rec_id)
    if not recording:
        return jsonify({'status': 'not found', 'rec_id': rec_id}), 404
    
    # IMPORTANT: Refresh the session to get updates from the worker thread
    db.session.refresh(recording)
    
    return jsonify({'status': recording.status, 'rec_id': rec_id})

@login_required 
def delete(db, Recording, rec_id):
    recording = db.session.get(Recording, rec_id)
    if not recording:
        flash('Recording not found.', 'error')
        return redirect(url_for('upload_and_list'))
        
    # Build file paths
    audio_path = os.path.join(current_app.config['UPLOAD_FOLDER'], recording.filename)
    transcript_path = os.path.join(current_app.config['TRANSCRIPT_FOLDER'], recording.filename + ".txt")
    
    # 1. Delete files
    if os.path.exists(audio_path):
        os.remove(audio_path)
    if os.path.exists(transcript_path):
        os.remove(transcript_path)
    
    # 2. Delete database entry
    db.session.delete(recording)
    db.session.commit()
    
    flash(f'Recording ID {rec_id} and associated files deleted.', 'success')
    return redirect(url_for('upload_and_list'))

@login_required 
def reprocess(db, Recording, model_key, rec_id):
    recording = db.session.get(Recording, rec_id)
    if not recording:
        flash('Recording not found.', 'error')
        return redirect(url_for('upload_and_list'))
        
    # Check if the requested model key is valid
    if model_key not in WHISPER_MODELS:
        flash(f'Invalid model key: {model_key}', 'error')
        return redirect(url_for('upload_and_list'))
        
    # Update the status and model
    recording.status = 'pending' # Reset to pending
    recording.model_key = model_key
    db.session.commit()
    
    # FIX: Explicitly resolve current_app to the actual object before passing to thread
    app_instance = current_app._get_current_object()
    start_transcription_worker(recording.id, model_key, app_instance, db, Recording)
    
    flash(f'Recording ID {rec_id} is being reprocessed with model: {model_key}.', 'info')
    return redirect(url_for('upload_and_list'))

@login_required 
def cleanup(db, Recording):
    # FIX: Explicitly resolve current_app to the actual object before passing to worker
    app_context = current_app._get_current_object() 
    # Get the paths from config attached to the current_app context
    UPLOAD_FOLDER = current_app.config['UPLOAD_FOLDER']
    TRANSCRIPT_FOLDER = current_app.config['TRANSCRIPT_FOLDER']
    
    clean_zombie_entries(app_context, db, Recording, UPLOAD_FOLDER, TRANSCRIPT_FOLDER)
    return redirect(url_for('upload_and_list'))


# --- Route Registration ---
def register_routes(app, db, Recording, User): # <-- Accepts User Model
    """Registers all routes with the Flask application instance."""
    
    # NEW LOGIN/LOGOUT ROUTES
    app.add_url_rule("/login", view_func=lambda: login(db, User), endpoint="login", methods=['GET', 'POST'])
    app.add_url_rule("/logout", view_func=logout, endpoint="logout")
    
    # CONDITIONAL REGISTRATION ROUTE - SECURED BY CONFIG.PY
    if REGISTRATION_ENABLED:
        app.add_url_rule("/register", view_func=lambda: register(db, User), endpoint="register", methods=['GET', 'POST']) 
    
    # PROTECTED ROUTES
    app.add_url_rule("/", view_func=lambda: upload_and_list(db, Recording), endpoint="upload_and_list", methods=['GET', 'POST'])
    app.add_url_rule("/transcript/<int:rec_id>", view_func=lambda rec_id: view_transcript(db, Recording, rec_id), endpoint="view_transcript")
    app.add_url_rule("/audio/<int:rec_id>", view_func=lambda rec_id: serve_audio(db, Recording, rec_id), endpoint="serve_audio")
    app.add_url_rule("/download/<int:rec_id>", view_func=lambda rec_id: download_transcript(db, Recording, rec_id), endpoint="download_transcript")
    app.add_url_rule("/status/<int:rec_id>", view_func=lambda rec_id: recording_status(db, Recording, rec_id), endpoint="recording_status")
    app.add_url_rule("/delete/<int:rec_id>", view_func=lambda rec_id: delete(db, Recording, rec_id), endpoint="delete")
    app.add_url_rule("/reprocess/<string:model_key>/<int:rec_id>", 
                     view_func=lambda model_key, rec_id: reprocess(db, Recording, model_key, rec_id),
                     endpoint="reprocess")
    app.add_url_rule("/cleanup", view_func=lambda: cleanup(db, Recording), endpoint="cleanup")