#!/usr/bin/env python3
import os
import sys
import shutil
import subprocess
from flask import flash # Import flash for use inside context

# We only import the configuration settings, which are independent.
from config import WHISPER_PATH, UPLOAD_FOLDER, TRANSCRIPT_FOLDER, BASE_DIR

# The function must now accept db and Recording models, and app context.
def transcribe_recording(rec_id: int, model_key: str, db_context, Recording_model, app_context):
    
    # 1. Determine model path based on key
    whisper_models_dir = os.path.expanduser("~/whisper.cpp/models")
    model_filename = f"ggml-{model_key}.bin"
    model_path = os.path.join(whisper_models_dir, model_filename)

    if not os.path.isfile(model_path):
        # NOTE: Using print() here as flash() requires app context
        print(f"ERROR: Model file not found at {model_path}. Falling back to default.")
        model_path = os.path.join(whisper_models_dir, "ggml-small.bin") # Fallback
        if not os.path.isfile(model_path):
             print(f"FATAL ERROR: Default model (ggml-small.bin) not found. Cannot proceed.")
             # No DB update possible here without context, just exit
             return 

    # Must run database operations within the application context
    with app_context.app_context():
        
        # We need to re-fetch the recording item within this new thread's context 
        item = db_context.session.get(Recording_model, rec_id)
        
        if not item:
            print(f"Worker: Recording ID {rec_id} not found in DB.")
            return

        print(f"Starting transcription/translation for ID {rec_id} ({item.filename}) using model {model_key}...")

        # 2. Update status to processing
        item.status = "processing"
        db_context.session.commit()
        
        # 3. Define file paths
        input_file = os.path.join(UPLOAD_FOLDER, item.filename)
        file_stem, _ = os.path.splitext(item.filename)
        output_cli_path = os.path.join(UPLOAD_FOLDER, file_stem)
        final_txt = os.path.join(TRANSCRIPT_FOLDER, file_stem + ".txt")


        # 4. Construct the whisper.cpp command
        command = [
            WHISPER_PATH,
            "-m", model_path,
            "-f", input_file,
            "-otxt",
            "-of", output_cli_path,
            "-l", "auto",
            "-p", "4",
            # THE FIX: ADD THIS FLAG TO TRANSLATE TO ENGLISH
            "-tr" 
        ]

        # 5. Execute the command
        try:
            result = subprocess.run(command, check=True, capture_output=True, text=True)
            print(f"Whisper CLI Output (stdout): {result.stdout}")
        except subprocess.CalledProcessError as e:
            print(f"Whisper CLI failed for ID {rec_id}: {e}")
            print(f"Stderr: {e.stderr}")
            item.status = "error: CLI Failed"
            db_context.session.commit()
            return
        except FileNotFoundError:
            print(f"Whisper CLI executable not found at {WHISPER_PATH}. Check config.py.")
            item.status = "error: CLI Not Found"
            db_context.session.commit()
            return
        except Exception as e:
            print(f"An unexpected error occurred during transcription: {e}")
            item.status = "error: Internal"
            db_context.session.commit()
            return
            
        # 6. Locate the output file
        final_cli_path = output_cli_path + ".txt"

        # 7. Sanity check: ensure the output file exists and is not empty
        if not os.path.exists(final_cli_path) or os.path.getsize(final_cli_path) == 0:
            print(f"ERROR: Transcript file not found or is empty after CLI execution: {final_cli_path}")
            item.status = "error: Empty Transcript"
            db_context.session.commit()
            return

        # 8. Move and RENAME transcript
        os.makedirs(TRANSCRIPT_FOLDER, exist_ok=True)
        shutil.move(final_cli_path, final_txt)

        # 9. Read and update database
        try:
            with open(final_txt, "r", encoding='utf-8') as f:
                transcript = f.read().strip()
                
            item.transcript = transcript
            item.status = "done"
            item.model_key = model_key 

        except Exception as e:
            print(f"Failed to read transcript: {e}")
            item.status = "error: Read Failed"
            db_context.session.commit()
            return

        # 10. Final commit for 'done' status
        db_context.session.commit()
        print(f"Transcription complete: {final_txt}")